#! /usr/bin/env R
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
message("Log file for code executed at\n")
message(format(Sys.time(), "%a %b %d %X %Y"))
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
library(devtools)

library(ggplot2)
library(ggsci)
library(grid)
library(scales)
library(gridExtra)
library(tikzDevice)
library(bit64);
library(magrittr);
library(stringr);
library(glue)
library(haven)
library(zeallot)

library(data.table);
library(statar)
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
latex_percent <- function (x) {
    x <- plyr::round_any(x, scales:::precision(x)/100)
    stringr::str_c(comma(x * 100), "\\%")
}
# ------------------------------------------------------------------------------------------


# ------------------------------------------------------------------------------------------
# var = "log_import"
prep_plot_DiD <- function(var, controls="gravity"){
  dt1 <- fread(glue("./output/coef_for_plots/coef_for_plot_postest_{controls}_{var}.txt"), skip=3)
  setnames(dt1, c("lag", "coef", "se", "pval"))
  dt1[, coef := as.numeric(coef) ]
  dt1 <- dt1[ !is.na(coef) & !(lag %in% c("N", "r2") ) ]
  dt1[, horizon := as.numeric(gsub("\\D", "", lag)) ]
  dt1 <- dt1[!is.na(horizon), .(year_diff=fifelse(str_sub(lag, 1, 1)=="l", -1, 1) * horizon, 
    coef=as.numeric(coef), se=as.numeric(se) ) ]
  return(dt1)
}
dt_plot_import <- prep_plot_DiD("win_log_import")
# dt_plot_import <- prep_plot_DiD("log_import")
dt_plot_R2     <- prep_plot_DiD("R2_y")
dt_plot_beta   <- prep_plot_DiD("beta")
dt_plot_fxvol  <- prep_plot_DiD("fx_vol")
dt_plot_voladj <- prep_plot_DiD("var_adj")
dt_plot_cls1   <- prep_plot_DiD("unshared_cls1")
dt_plot_cls2   <- prep_plot_DiD("unshared_cls2")

# --- plots 
make_plot_DiD <- function(dt_plot_var, 
  xlab="", ylab="", 
  se_mult=2.0, 
  horizon=10){
  p1 <- dt_plot_var[ abs(year_diff) <= horizon ] %>%
      ggplot(aes(x=year_diff, y=coef)) + 
      geom_line(alpha=0.55) + geom_point(shape=1, alpha = 1.0, colour="black") + 
      geom_errorbar(aes(ymin=coef-se_mult*se, ymax=coef+se_mult*se), 
                    colour="black", alpha=0.5, width=0.15) +
      geom_vline(xintercept=0, color="red", linetype="dashed") + 
      geom_hline(yintercept=0, color="black", linetype="solid") + 
      theme_bw() + 
      theme(axis.text = element_text(size=8), ) + 
       scale_y_continuous(labels = latex_percent) + 
       # scale_y_continuous(labels = scales::percent) + 
      xlab(xlab) + ylab(ylab) +
      theme(axis.title=element_text(size=9))
  return(p1)    
}

# - trade
dt_plot_var <- copy(dt_plot_import)
p_trade <- make_plot_DiD(dt_plot_import, xlab="Years from Trade Deal", se_mult=2.0, horizon=10)

p_R2 <- make_plot_DiD(dt_plot_R2)
my_grob = grobTree(textGrob("R-squared, $R^2$", x=0.02,  y=0.2, hjust=0,
  gp=gpar(col="black", fontsize=8)))
p_R2 <- p_R2 + annotation_custom(my_grob)
p_beta <- make_plot_DiD(dt_plot_beta)
my_grob = grobTree(textGrob("Base Loading, $\\beta$", x=0.02,  y=0.2, hjust=0,
  gp=gpar(col="black", fontsize=8)))
p_beta <- p_beta + annotation_custom(my_grob)

p_fxvol <- make_plot_DiD(dt_plot_fxvol)
my_grob = grobTree(textGrob("Volatility, $\\sigma$", x=0.02,  y=0.2, hjust=0,
  gp=gpar(col="black", fontsize=8)))
p_fxvol <- p_fxvol + annotation_custom(my_grob)
p_voladj <- make_plot_DiD(dt_plot_voladj)
my_grob = grobTree(textGrob("Unshared risk, $\\rho$", x=0.02,  y=0.2, hjust=0,
  gp=gpar(col="black", fontsize=8)))
p_voladj <- p_voladj + annotation_custom(my_grob)

p_cls1 <-  make_plot_DiD(dt_plot_cls1)
p_cls2 <-  make_plot_DiD(dt_plot_cls2)


# --- figure 1: output in tikz / eps / pdf
tikz("./output/figures/figure1.tex", width=5, height=3)
  print(p_trade)
dev.off()
ggsave("./output/figures/figure1.pdf", p_trade, width=5, height=3, units="in")

# --- postscript for publication 
# --- commented out because this depends on your cairo setup
# ggsave(filename = "./output/figures/figure1.eps",
#        plot = print(p_trade),
#        width=5, height=3, units="in",
#        device = cairo_ps)
# ggsave(filename = "./output/figures/figure1.ps",
#        plot = print(p_trade),
#        width=5, height=3, units="in",
#        device = cairo_ps)



# --- figure 2a and 2b
tikz( "./output/figures/figure2a.tex", width=7, height=3)
  grid.arrange(p_fxvol, p_voladj, nrow = 1)
dev.off()
ggsave("./output/figures/figure2b.pdf", 
       grid.arrange(p_fxvol, p_voladj, nrow = 1), 
       width=7, height=3, units="in")
# --- postscript for publication 
# ggsave(filename = "./output/figures/figure2a.eps", 
#        plot = grid.arrange(p_fxvol, p_voladj, nrow = 1),
#        width=7, height=3, units="in",
#        device = cairo_ps)
# ggsave(filename = "./output/figures/figure2a.ps",
#        plot = grid.arrange(p_fxvol, p_voladj, nrow = 1),
#        width=7, height=3, units="in",
#        device = cairo_ps)


tikz( "./output/figures/figure2b.tex", width=7, height=3)
  grid.arrange(p_R2, p_beta, nrow = 1)
dev.off()
ggsave("./output/figures/figure2b.pdf", 
       grid.arrange(p_fxvol, p_voladj, nrow = 1), 
       width=7, height=3, units="in")
# --- postscript for publication 
# ggsave(filename = "./output/figures/figure2b.eps", 
#        plot = grid.arrange(p_fxvol, p_voladj, nrow = 1),
#        width=7, height=3, units="in",
#        device = cairo_ps)
# ggsave(filename = "./output/figures/figure2b.ps",
#        plot = grid.arrange(p_fxvol, p_voladj, nrow = 1),
#        width=7, height=3, units="in",
#        device = cairo_ps)


# ------------------------------------------------------------------------------------------










